//
//  nanoTimer.swift
//  NanoTimer
//
//  Created by Guan Gui on 14/09/2014.
//  Copyright (c) 2014 Guan Gui. All rights reserved.
//

import Foundation

struct NanoTimer {
    private static var t = mach_timebase_info(numer: 0, denom: 0)
    private var timeElapsed = UInt64(0), lastLap = Int64(0)
    private var startTime = UInt64(0)
    var ownCost = UInt64(0)
    
    var minutes: Double {
        return Double(duration) / 60_000_000_000
    }
    
    var seconds: Double {
        return Double(duration) / 1_000_000_000
    }
    
    var milliseconds: Double {
        return Double(duration) / 1_000_000
    }
    
    var microseconds: Double {
        return Double(duration) / 1_000
    }
    
    var nanoseconds: Double {
        return Double(duration)
    }
    
    var duration: UInt64 {
        return machAbsoluteTimeToNanoseconds(timeElapsed)
    }
    
    var lap: UInt64 {
        return machAbsoluteTimeToNanoseconds(lastLap > 0 ? UInt64(lastLap) : 0)
    }
    
    init() {
        if NanoTimer.t.denom == 0 {
            mach_timebase_info(&NanoTimer.t)
        }
    }
    
    mutating func start() {
        startTime = mach_absolute_time()
    }
    
    mutating func stop() {
        let delta = mach_absolute_time() - startTime
        calibrateOwnCost(numOfRun: 1)
        lastLap = Int64(delta) - Int64(ownCost)
        if lastLap < 0 {
            let absLastLap = UInt64(-lastLap)
            if timeElapsed < absLastLap {
                timeElapsed = 0
            } else {
                timeElapsed -= absLastLap
            }
        } else {
            let absLastLap = UInt64(lastLap)
            timeElapsed += absLastLap
        }
    }
    
    mutating func reset() {
        startTime = 0
        timeElapsed = 0
        lastLap = 0
    }
    
    private func machAbsoluteTimeToNanoseconds(machAbsoluteTime: UInt64) -> UInt64 {
        return (machAbsoluteTime * UInt64(NanoTimer.t.numer)) / UInt64(NanoTimer.t.denom)
    }
    
    mutating func calibrateOwnCost(numOfRun: UInt64 = 100) {
        var startTime: UInt64
        var timeElapsed = UInt64(0)
        for _ in 0..<numOfRun {
            startTime = mach_absolute_time()
            timeElapsed += mach_absolute_time() - startTime
        }
        ownCost = timeElapsed / numOfRun
    }
}